
<?php
// Simple dashboard UI. Drop this in /public
?>
<!doctype html>
<html>
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Cabropex Tower Monitor</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <style>
    :root { --bg:#0f172a; --card:#111827; --muted:#94a3b8; --text:#e5e7eb; --accent:#22d3ee; }
    * { box-sizing: border-box; }
    body { margin:0; font-family:Inter, system-ui, -apple-system; background:#0b1220; color:var(--text); }
    header { padding:16px 24px; border-bottom:1px solid #1f2937; display:flex; align-items:center; gap:12px; }
    .logo { width:12px; height:12px; border-radius:50%; background:var(--accent); display:inline-block; }
    .wrap { padding:20px; max-width:1200px; margin:0 auto; }
    .grid { display:grid; grid-template-columns: repeat(12, 1fr); gap:16px; }
    .card { background:#0f1629; border:1px solid #1f2937; border-radius:16px; padding:16px; }
    .span-3{ grid-column: span 3; } .span-4{ grid-column: span 4; } .span-6{ grid-column: span 6; } .span-8{ grid-column: span 8; } .span-12{ grid-column: span 12; }
    .title { font-weight:700; font-size:14px; color:#cbd5e1; margin-bottom:10px; text-transform:uppercase; letter-spacing:.08em; }
    .status { display:flex; gap:16px; flex-wrap:wrap; }
    .pill { background:#0b1325; border:1px solid #1f2937; padding:8px 12px; border-radius:999px; font-size:13px; color:#cbd5e1; }
    .ok{ color:#10b981; } .warn{ color:#f59e0b; } .alarm{ color:#ef4444; }
    small{ color: var(--muted); }
    .kv{ display:grid; grid-template-columns: 1fr auto; gap:6px; font-size:14px; }
    .kv div:nth-child(odd){ color:#9fb2d0; }
  </style>
</head>
<body>
<header>
  <span class="logo"></span>
  <h1 style="font-size:16px; margin:0;">Cabropex Tower Monitor</h1>
</header>

<div class="wrap">
  <div class="grid">
    <div class="card span-12">
      <div class="title">Site Status</div>
      <div class="status" id="status">
        <span class="pill" id="grid-pill">Grid: <b>--</b></span>
        <span class="pill" id="gen-pill">Generator: <b>--</b></span>
        <span class="pill" id="solar-pill">Solar: <b>--</b></span>
        <span class="pill" id="batt-pill">Battery SOC: <b>--</b></span>
        <span class="pill" id="dc-pill">DC: <b>--</b></span>
      </div>
      <small>Site: <span id="site-name">site001</span> — last refresh <span id="last-refresh">--</span></small>
    </div>

    <div class="card span-4">
      <div class="title">Generator</div>
      <div class="kv" id="gen-kv">
        <div>Status</div><div id="gen-status">--</div>
        <div>Mode</div><div id="gen-mode">--</div>
        <div>Fuel Level</div><div id="gen-fuel">--</div>
        <div>Engine Hours</div><div id="gen-hours">--</div>
      </div>
    </div>

    <div class="card span-4">
      <div class="title">AC Grid</div>
      <div class="kv">
        <div>Grid</div><div id="grid-status">--</div>
        <div>V (L1/L2/L3)</div><div id="grid-v">--</div>
        <div>I (L1/L2/L3)</div><div id="grid-i">--</div>
        <div>Power (kW)</div><div id="grid-p">--</div>
      </div>
    </div>

    <div class="card span-4">
      <div class="title">Battery / DC</div>
      <div class="kv">
        <div>SOC / SOH</div><div id="batt-soc">--</div>
        <div>Voltage / Current</div><div id="batt-vi">--</div>
        <div>Temp (°C)</div><div id="batt-t">--</div>
        <div>DC Bus</div><div id="dc-vi">--</div>
      </div>
    </div>

    <div class="card span-12">
      <div class="title">Trends (last 24h)</div>
      <canvas id="chart1" height="110"></canvas>
    </div>
  </div>
</div>

<script>
const SITE = new URLSearchParams(location.search).get('site') || 'site001';
document.getElementById('site-name').textContent = SITE;

async function fetchLatest(){
  const res = await fetch('../api/latest.php?site_id=' + encodeURIComponent(SITE));
  return await res.json();
}
async function fetchHistory(stream, hours=24){
  const res = await fetch('../api/history.php?site_id=' + encodeURIComponent(SITE) + '&stream=' + stream + '&hours=' + hours);
  return await res.json();
}

function setText(id, val){ document.getElementById(id).textContent = val; }

function setPill(id, text, cls){
  const el = document.getElementById(id);
  el.innerHTML = text;
  el.classList.remove('ok','warn','alarm');
  if(cls) el.classList.add(cls);
}

function fmt(n){ if(n===null || n===undefined) return '--'; return Number(n).toFixed(1); }

async function refresh(){
  const data = await fetchLatest();
  const ac = data.ac || {}; const gen = data.generator || {};
  const batt = data.battery || {}; const dc = data.dc || {}; const solar = data.solar || {};

  // Status row
  setPill('grid-pill', 'Grid: <b>' + (ac.grid_status || '--') + '</b>', ac.grid_status==='ON'?'ok':'alarm');
  setPill('gen-pill', 'Generator: <b>' + (gen.status || '--') + '</b>', gen.status==='ON'?'warn':'');
  setPill('solar-pill', 'Solar: <b>' + fmt(solar.power_kw) + ' kW</b>', 'ok');
  setPill('batt-pill', 'Battery SOC: <b>' + fmt(batt.soc_pct) + '%</b>', (batt.soc_pct<30)?'alarm':(batt.soc_pct<50)?'warn':'ok');
  setPill('dc-pill', 'DC: <b>' + fmt(dc.voltage_v) + ' V</b>');

  // Generator card
  setText('gen-status', gen.status || '--');
  setText('gen-mode', gen.mode || '--');
  setText('gen-fuel', (gen.fuel_level_pct!==undefined? fmt(gen.fuel_level_pct)+'%':'--'));
  setText('gen-hours', (gen.engine_hours!==undefined? fmt(gen.engine_hours)+' h':'--'));

  // AC card
  const gv = [ac.v_l1, ac.v_l2, ac.v_l3].map(fmt).join('/');
  const gi = [ac.i_l1, ac.i_l2, ac.i_l3].map(fmt).join('/');
  setText('grid-status', ac.grid_status || '--');
  setText('grid-v', gv);
  setText('grid-i', gi);
  setText('grid-p', fmt(ac.power_kw));

  // Battery / DC
  setText('batt-soc', fmt(batt.soc_pct) + '% / ' + fmt(batt.soh_pct) + '%');
  setText('batt-vi', fmt(batt.voltage_v) + ' V / ' + fmt(batt.current_a) + ' A');
  setText('batt-t', fmt(batt.temp_c));
  setText('dc-vi', fmt(dc.voltage_v) + ' V / ' + fmt(dc.current_a) + ' A');

  document.getElementById('last-refresh').textContent = new Date().toLocaleTimeString();
}

let chart;
async function drawChart(){
  const battHist = await fetchHistory('battery', 24);
  const acHist = await fetchHistory('ac', 24);
  const labels = (battHist||[]).map(r => new Date(r.ts).toLocaleTimeString());
  const soc = battHist.map(r => Number(r.soc_pct));
  const power = acHist.map(r => Number(r.power_kw));
  const ctx = document.getElementById('chart1').getContext('2d');
  if (chart) chart.destroy();
  chart = new Chart(ctx, {
    type: 'line',
    data: {
      labels,
      datasets: [
        { label: 'Battery SOC %', data: soc, yAxisID: 'y1' },
        { label: 'AC Power kW', data: power, yAxisID: 'y2' }
      ]
    },
    options: {
      responsive: true,
      interaction: { mode: 'index', intersect: false },
      scales: { y1: { type:'linear', position:'left' }, y2: { type:'linear', position:'right' } }
    }
  });
}

refresh(); drawChart();
setInterval(refresh, 5000);
</script>
</body>
</html>
